%% Time-lapse video recovery using joint time-vertex regularizer
%
%   In this demo, we perform inpainting of time-lapse video using joint
%   time-vertex regularizer and we compare its performance with baseline
%   methods based on 3D Tikhonov and 3D Total Variation.


% Author: Francesco Grassi
% Date: April 2017

clear
close all
clc

gsp_start;
init_unlocbox;


compute_error = @(x,y) norm(vec(x-y),'fro')/norm(vec(x),'fro');


%% Load video


v = VideoReader('KLCC.mp4');
t = 0;
while hasFrame(v)
    t = t+1;
    video(:,:,:,t) = readFrame(v);
end


%% Signal and Corrupted video
X = video(1:3:end,1:3:end,:,:);
[height,width,~,T] = size(X);

x = double(reshape(X,[],3,T));
xr = squeeze(x(:,1,:));
xg = squeeze(x(:,2,:));
xb = squeeze(x(:,3,:));

Y = X;
y = double(reshape(Y,[],3,T));

perc = 0.2; %percentage of missing pixels
M  = rand(height*width,1)>perc;
M  = repmat(M,1,T);
M(:,rand(1,T)<perc) = 0;

yr = squeeze(y(:,1,:));
yg = squeeze(y(:,2,:));
yb = squeeze(y(:,3,:));


yr = M.*yr;
yg = M.*yg;
yb = M.*yb;

y(:,1,:) = yr;
y(:,2,:) = yg;
y(:,3,:) = yb;

compute_error(x,y)


%% Graph
ymean = mean(y,3);
param.nnparam.k = 4;
param.patch_size = 1;
param.rho = 700;
param.transform = 'dct';
G = gsp_patch_graph(reshape(ymean,height,width,3),param);
G = gsp_update_weights(G,G.W/max(G.W(:)));
G = gsp_adj2vec(G);
G = gsp_estimate_lmax(G);
G = gsp_jtv_graph(G,T,1,param);

verbose = 1;
param_solver.maxit  = 300;
param_solver.verbose = verbose;

param_tv.use_matrix = 1;
param_tv.maxit      = 20;
param_tv.verbose    = verbose-1;
%% Solving Problems
joint12
joint11
joint21
joint22
tv_tik_baseline

%% Error for each time instant
for t = 1:T
    error(t)    = compute_error(x(:,:,t),y(:,:,t));
    errtik(t)   = compute_error(x(:,:,t),reshape(tik(:,:,:,t),[],3));
    errtv(t)    = compute_error(x(:,:,t),reshape(tv(:,:,:,t),[],3));
    errjoint12(t) = compute_error(x(:,:,t),z12(:,:,t));
    errjoint11(t) = compute_error(x(:,:,t),z11(:,:,t));
    errjoint21(t) = compute_error(x(:,:,t),z21(:,:,t));
    errjoint22(t) = compute_error(x(:,:,t),z22(:,:,t));
end


%%
err = [errtik,errtv,errjoint11,errjoint12,errjoint21,errjoint22];
grp = vec(repmat((1:6),numel(errtik),1));

figure
boxplot(err,grp,'colors','k','sym','k+')
set(gca,'XTickLabel',{'Tik','TV','N_{11}','N_{12}','N_{21}','N_{22}'})
set(gca,'TickLabelInterpreter', 'tex');

frameind = find(sum(M)==0);
pixelind = find(sum(M)>0);

%mean error different methods: pixels only, frame only, whole video
table_result = [mean(errtik(pixelind))     mean(errtik(frameind))     mean(errtik)    
mean(errtv(pixelind))      mean(errtv(frameind))      mean(errtv)
mean(errjoint11(pixelind)) mean(errjoint11(frameind)) mean(errjoint11)
mean(errjoint12(pixelind)) mean(errjoint12(frameind)) mean(errjoint12)
mean(errjoint21(pixelind)) mean(errjoint21(frameind)) mean(errjoint21)
mean(errjoint22(pixelind)) mean(errjoint22(frameind)) mean(errjoint22)];

