%% Localization of seismic epicenter using earthquake waveforms and STVWT
%
%   In this demo, we show how to estimate the epicenter of a seismic event
%   using the waveforms recorded by a seismic sensor network in New
%   Zealand. 
%   Dataset can be found at http://info.geonet.org.nz/display/appdata/Continuous+Waveform+Buffer
%

% Author: Francesco Grassi
% Date: April 2017

close all
clear
clc

gsp_start;
init_unlocbox;

folder = ['.\CWB\'];
addpath('.\libraries');
addpath(folder)

%% Parameters
param.k = 6;
param.use_full = 1;

T = 15e3;

alpha = linspace(0.1,2,8);
beta  = 0.01;

%Use elevation when build graph
elev=0;

%% Analysis
n = 0;
load listID.mat

for n_id = 1:numel(listID)
    
    fprintf('Importing listID: %s\n', listID{n_id})
    
    load([folder listID{n_id} '\' listID{n_id} '.mat'])
    
    %Select stations with recording length>T
    
    S = S([S.length]>=T);
    
    %Coordinates
    if elev
        coords = cell2mat({S.coordinates}');
    else
        coords = cell2mat({S.coordinates}');
        coords = coords(:,1:2);
    end
    
    nwf = numel(S);
    
    if nwf<80
        fprintf('Insufficient number of waveforms.\n')
        continue
    end
    
    X    = zeros(nwf,T);
    time = S(1).data(1:T,1);
    
    for ii=1:nwf
        X(ii,:)=S(ii).data(1:T,2)';
    end
    
    keep   = sum(abs(X).^2,2)<2*std(sum(abs(X).^2,2));
    X      = X(keep,:);
    coords = coords(keep,:);
    
    
    %Preprocessing
    X = X - mean(X(:));
    X = X - repmat(mean(X,2),1,T);
    X = X(:,1:2:end);
    X = X./max(abs(X(:)));
    energy = norm(X(:),'fro');
    
    T = size(X,2);
    
    %Graph creation
    G = gsp_nn_graph(coords,param);
    G = gsp_update_weights(G,double(G.W>0));
    
    G = gsp_jtv_graph(G,T,1);
    G = gsp_compute_fourier_basis(G);
    
    [~,origin_ind] = min(pdist2([E.longitude E.latitude],G.coords));
    
    fprintf('Number of stations N = %d \nSignals length T = %d\n',G.N,G.jtv.T);
    graph_dist = lldistkm(G.coords(origin_ind,2:-1:1),[E.latitude E.longitude]);
    
    
    %% Wave dictionary and analysis
    n = n+1;
    
    fprintf('Computing the filterbank.\n')
    [g,ft] = gsp_jtv_design_damped_wave(G,alpha,beta);
    gdual  = gsp_jtv_design_can_dual(g,ft);
    ge     = gsp_jtv_filter_evaluate(g,ft,G.e,gsp_jtv_ta(G,0));
    gduale = gsp_jtv_filter_evaluate(gdual,ft,G.e,gsp_jtv_ta(G,0));
    ge     = squeeze(mat2cell(ge,G.N,G.jtv.T,ones(1,numel(g))));
    gduale = squeeze(mat2cell(gduale,G.N,G.jtv.T,ones(1,numel(g))));
    
    A = @(x) gsp_jtv_filter_analysis(G,ge,ft,X,param);
    At = @(c) gsp_jtv_filter_synthesis(G,gduale,ft,c,param);
    
    
    tau(n)  = energy/G.N;
    coeff  = get_l1_dec(X,A,At,tau(n));
    
    while norm(coeff(:))<0.01
        tau(n)  = tau(n)-tau(n)*0.1;
        coeff  = get_l1_dec(X,A,At,tau(n));
    end
    
    fprintf('Parameters extraction\n')

    IDs(n)  = listID(n_id);
    weights = sum(sum(max(abs(coeff(:,1:5000,:)),[],3),3),2);

    origin_signal(n,:) = sum(abs(X(:,1:5000)),2)'*G.coords/sum(sum(abs(X(:,1:5000)),2));
    origin_estim(n,:)  = (weights'*G.coords)/sum(weights);
    
    dsignal(n,1) = lldistkm(origin_signal(n,2:-1:1),[E.latitude E.longitude]);
    destim(n,1)  = lldistkm(origin_estim(n,2:-1:1),[E.latitude E.longitude]);

    fprintf('%d beta %.2f source: signal %.3f filter %.3f\n',n, beta,dsignal(n),destim(n))
    
end

%% Summary statistics

boxplot([dsignal destim],'sym','+','color','k')
set(gca,'XTickLabel',{'Signal Amplitude' 'STVWT'})
set(gca,'Fontsize',13)
ylabel('Distance error [km]','Fontsize',13)

