%% Fast joint time-vertex filtering using FFC algorithm
%
%   In this demo, we perform fast joint time-vertex filtering using Fast
%   Fourier-Chebyshev algorithm using two joint filters: a separable
%   low-pass filter and a non-separable wave filter. Ideal filters are
%   approximated with smooth analytical functions.
%	To make the comparison with other methods the code was provided confidentially by the authors of 
%	E. Isufi, A. Loukas, A. Simonetto, and G. Leus, Autoregressivemoving  average  graph  filtering, IEEE  Trans.  Signal  Process.,
%

% Author: Francesco Grassi
% Date: April 2017
clear; clc;
close all

gsp_start;

N = 5000;
T = 1000;

G = gsp_sensor(N);
G = gsp_create_laplacian(G, 'normalized');
G = gsp_compute_fourier_basis(G);
G = gsp_jtv_graph(G, T);

omega = fftshift(G.jtv.omega);
lower = [ min( G.e ), min( omega ) ];
upper = [ max( G.e ), max( omega ) ];

[ X, Y ] = meshgrid( omega, G.e ); xy = [ Y(:) X(:) ];

%% Lowpass filter
r = 100;
M1 = [1 5:5:40]; %approximation orders
lcut = 1;
wcut = 0.25;

hlp = @( lambda,omega ) double(and(abs(lambda)<1,abs(omega)<wcut));
Hlp = hlp( Y,X );

%Approximation of Ideal Lowpass filter
hlp_approx = @(l,w) (1-1./(1+exp(-r*(l-lcut)))) .* (1-1./(1+exp(-r*(abs(w)-wcut))));
Hlp_approx =  hlp_approx( Y,X );


% Lowpass Filter approximation using FFC
param.method = 'cheby';
for m=1:numel(M1)
    param.order = M1(m); 
	Hlp_ffc = gsp_jft(G,gsp_jtv_filter_analysis(G,hlp_approx,'js',gsp_ijft(G,ones(G.N,G.jtv.T)),param));
    Hlp_ffc = fftshift(abs(Hlp_ffc),2);
    err_lowpass(m) = norm(Hlp-Hlp_ffc,'fro')/norm(Hlp,'fro');
end


%% Wave filter
r = 100;
M2 = [1 5:5:60];

hwave = @( l,w ) (exp(-abs(2*pi*abs(w)-acos(1-l/G.lmax/2)).^2*r))*sqrt(T)/2;
Hwave =  hwave( Y,X );
err_wave = zeros(numel(M),1);

param.method = 'cheby';
for m=1:numel(M2)
    param.order = M2(m);
    Hwave_ffc = fftshift(abs(gsp_jft(G,gsp_jtv_filter_analysis(G,hwave,'js',gsp_ijft(G,ones(G.N,G.jtv.T)),param))),2);
    err_wave(m) = norm(Hwave-Hwave_ffc,'fro')/norm(Hwave,'fro');
end

%%
figure
subplot(121)
semilogy(M1,err_lowpass,'k.-')
xlabel('Order')
ylabel('Normalized error')
title('Lowpass filter')

subplot(122)
semilogy(M2,err_wave,'k.-')
xlabel('Order')
ylabel('Normalized error')
title('Wave filter')