%% Joint Harmonic Analysis of Epidemic Spreading and compactness analysis
%
%   In this demo, we simulate an epidemic spreading using SEIR and SEIRS
%   compartmental model, we show JFT for different parameters, illustrating
%   how it can be used to discriminate between models.
%   Finally, we compute compactness for different representations and show
%   results.
%

% Author: Francesco Grassi
% Date: April 2017

close all
clear; clc;

load 'network.mat';

gsp_start;
%% Create connectivity graph
% Select nodes corresponding only to Europe
sample = and(c(:,1)>-31.3115 ,c(:,1)<  61.7221);
sample = sample.*and(c(:,2)> 20.4743,c(:,2)<  80.2162);
sample = logical(sample);
c = c(sample,:);

% Create k-nn graph and take biggest component
paramnn.k = 6;
G = gsp_nn_graph(c,paramnn);
[idx,Gz] = gsp_components(G);
[~, ind] = max(grpstats(idx,idx,'numel'));
G = Gz{ind};

%Add graph of flight routes
A = G.W>0;
Afly = Afly(sample,sample);
Afly = Afly(idx==1,idx==1)/100;
A = A+Afly;
G = gsp_update_weights(G,A);
G = gsp_compute_fourier_basis(G);

%% Simulate epidemic spreading
Tim = [20 20 Inf Inf];
cp  = [0.004 0.03 0.004 0.03];

for ii = 1:4
    contagion_prob    = cp(ii);            % contagion probability per day
    infection_length  = 6;                % length of infection period (days)
    params.maxTime    = round(16*365/12);    % days (before cure is found)
    params.population = 70;               % number of people in each airport (used to attain higher precision that 0/1). Don't increase too much as it takes longer.
    params.model      = 'SEIRS';             % infection model (SI, SIR)
    params.immunity   = Tim(ii);
    params.latency    = 2;
    patient_zero      = randi(size(A,1),3,1);
    
    % run the model
    [S, I, R, E] = compartmental_epidemics(sparse(A), contagion_prob, infection_length, patient_zero, params);
    fprintf('infection complete!\n');
    
    I   = I - mean(I(:));
    X{ii} = I;
end

%% Show results
ha = [];
for ii=1:4
    size(X{ii},2)
    G = gsp_jtv_graph(G,size(X{ii},2));
    Ihat = gsp_jft(G,X{ii});
    subplot(2,2,ii)
    ha(ii) = gca;
    param.logscale = 1;
    param.dB = Inf;
    gsp_plot_jft(G,Ihat,param)
    ylim([0 7]);
    xlim([-0.25 0.25]);
    xlabel('Frequency [\omega]')
    ylabel('Eigenvalue [\lambda]')
end
title(ha(1),'Low Probability')
title(ha(2),'High Probability')

colormap(flipud(gray))
ht = text(-1.05,2,'SEIRS');
set(ht,'Rotation',90)
ht = text(-1.05,13,'SEIR');
set(ht,'Rotation',90)

%% Measure compactness

ind = 1;
G = gsp_jtv_graph(G,size(X{ind},2));
p = 0:5:100;

Xhat = gsp_jft(G,X{ind});
Xdot = gsp_tft(G,X{ind});
Xtil = gsp_gft(G,X{ind});

for ii=1:numel(p)
    Xhat2 = Xhat;
    Xdot2 = Xdot;
    Xtil2 = Xtil;
    
    Xhat2(abs(Xhat2)<prctile(abs(Xhat2(:)),p(ii))) = 0;
    Xdot2(abs(Xdot2)<prctile(abs(Xdot2(:)),p(ii))) = 0;
    Xtil2(abs(Xtil2)<prctile(abs(Xtil2(:)),p(ii))) = 0;
    
    err_jft(ii) = norm(vec(X{ind} - gsp_ijft(G,Xhat2)),'fro')/norm(vec(X{ind}),'fro');
    err_fft(ii) = norm(vec(X{ind} - gsp_itft(G,Xdot2)),'fro')/norm(vec(X{ind}),'fro');
    err_gft(ii) = norm(vec(X{ind} - gsp_igft(G,Xtil2)),'fro')/norm(vec(X{ind}),'fro');
end

figure
semilogy(p,err_jft,'k.-',p,err_fft,'ko-',p,err_gft,'k--','linewidth',1)
ylim([1e-4 1])
legend('JFT','DFT','GFT')
xlabel('percentile of removed entries')
ylabel('normalized error')
grid on

